#ifndef __TAiffFileFormat__
#define __TAiffFileFormat__

#include <Host/CEndian.hpp>
#include "IAudioFileFormat.hpp"
using Exponent::Audio::IAudioFileFormat;
using Exponent::Host::CEndian;

/**
 * @def DOUBLE_TO_UNSIGNED_LONG
 * @brief Convert a double to an unsigned long
 */
#define DOUBLE_TO_UNSIGNED_LONG(f)((unsigned long)(((long)(f - 2147483648.0)) + 2147483647L) + 1)

//	===========================================================================

namespace Exponent
{
	namespace Audio
	{
		/**
		 * @class TAiffFileFormat TAiffFileFormat.hpp
		 * @brief Specifies the aiff file format
		 *
		 * @date 13/04/2006
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 * @see IAudioFormat
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * @note Documented online at <A HREF="http://www.borg.com/~jglatt/tech/aiff.htm">http://www.borg.com/~jglatt/tech/aiff.htm</A>
		 *
		 * $Id: TAiffFileFormat.hpp,v 1.6 2007/02/08 21:08:09 paul Exp $
		 */
		template<typename TypeName> class TAiffFileFormat : public IAudioFileFormat<TypeName>
		{
		public:

			/**
			 * Construction
			 */
			TAiffFileFormat()
			{
				m_bitDepth = IAudioFileFormat<TypeName>::e_sixteenBit;
			}

			/**
			 * Destruction
			 */
			virtual ~TAiffFileFormat()
			{
				m_stream.closeStream();
			}

//	===========================================================================

			/**
			 * Open the file
			 * @param mode The mode of opening, read or write
			 * @param filename The name of the file to open
			 * @retval bool True if stream is opened
			 */
			virtual bool openFile(CFileStream::EStreamMode mode, const CSystemString &filename)
			{
				return m_stream.openStream(filename, mode, false);
			}

			/**
			 * Close the file
			 * @retval bool True if closed properly, false otherwise
			 */
			virtual bool closeFile()
			{
				m_stream.closeStream();
				return true;
			}

//	===========================================================================

			/**
			 * Get the extension of the file format, each array entry sould contain one string
			 * @param array The array to fill in
			 */
			virtual void getFileExtension(TStringCountedPointerArray &array) const
			{
				array.addElement(new CString("aif"));
				array.addElement(new CString("aiff"));
				array.addElement(new CString("aifc"));
			}

			/**
			 * Check if a file is a valid format of this file format
			 * @param filename The name of the file to load
			 * @retval bool True if the file format matches (checks actual binary data, not just extension
			 */
			virtual bool isValidFormat(const CSystemString &filename) const
			{
				// Create the stream
				CFileStream stream(filename, CFileStream::e_input, true);

				// Check open
				if (!stream.isStreamOpen())
				{
					return false;
				}

				// Read in the header
				typename TAiffFileFormat<TypeName>::SAiffFileFormat diskFormat;
				return readFileHeader(stream, diskFormat);
			}

//	===========================================================================

			/**
			 * Read the header information and store a copy in the format
			 * @param format On return is filled with the values for this audio file
			 * @retval bool True if read the header correctly, false otherwise
			 */
			virtual bool readHeader(typename IAudioFileFormat<TypeName>::SAudioFileFormat &format)
			{
				// Read in the header
				typename TAiffFileFormat<TypeName>::SAiffFileFormat diskFormat;
				if (!readFileHeader(m_stream, diskFormat))
				{
					return false;
				}
				// STore the output information
				format.m_sampleRate		  = diskFormat.m_sampleRate;
				format.m_numberOfChannels = diskFormat.m_numberOfChannels;
				format.m_numberOfSamples  = diskFormat.m_numberOfSampleFrames;
				format.m_bitDepth		  = (unsigned long)diskFormat.m_bitsPerSample;
				m_bitDepth				  = (typename IAudioFileFormat<TypeName>::EBitDepth)format.m_bitDepth;

				// Done :)
				return true;
			}

			/**
			 * Read the entire audio file to the provided audio buffer
			 * @param buffer The data buffer to store te samples in
			 * @retval bool True if read correctly, false otherwise
			 */
			virtual bool readData(TAudioBuffer< TypeName > &buffer)
			{
				//m_stream.setStreamEndianess(true);

				// Handle based upon the depth
				switch(m_bitDepth)
				{
					case IAudioFileFormat<TypeName>::e_eightBit:
						{
							// Store the entire size of the buffer to be read in
							const unsigned long size = buffer.getBufferSize();

							// Pointer to the data that we will iterate through
							TypeName *data = buffer.getMutableData();

							// The sample we read in to
							//unsigned char sample = 0;
							unsigned char *sample = new unsigned char[size];

							// Read the audio data
							if (!m_stream.readDataFromStream(sample, size))
							{
								FREE_ARRAY_POINTER(sample);
								return false;
							}

							// Save doing multiple divisions...
							const TypeName inversion = (TypeName)(1.0 / 255.0);

							// For the number of samples, we want to read in and convert to floating point range..
							for (unsigned long i = 0; i < size; i++)
							{
								//m_stream >> sample;
								//*data++ = (TypeName)((((TypeName)sample / (TypeName)255) * 2.0) - 1.0);
								*data++ = (TypeName)((((TypeName)sample[i] * inversion) * 2.0) - 1.0);
							}

							// Delete the buffer we loaded with
							FREE_ARRAY_POINTER(sample);
						}
					break;
					case IAudioFileFormat<TypeName>::e_sixteenBit:
						{
							// Store the entire size of the buffer to be read in
							const unsigned long size = buffer.getBufferSize();

							// Pointer to the data that we will iterate through
							TypeName *data = buffer.getMutableData();

							// The sample we read in to
							short *sample = new short[size];

							// Read the audio data
							if(!m_stream.readShortsFromStream(sample, size))
							{
								FREE_ARRAY_POINTER(sample);
								return false;
							}

							// Save doing multiple divisions...
							const TypeName inversion = (TypeName)(1.0 / 32768.0);

							// For the number of samples, we want to read in and convert to floating point range..
							for (unsigned long i = 0; i < size; i++)
							{
								*data++ = (TypeName)sample[i] * inversion;
							}

							// Delete the sample buffer...
							FREE_ARRAY_POINTER(sample);
						}
					break;
					case IAudioFileFormat<TypeName>::e_twentyFourBit:
						{
							// Store the entire size of the buffer to be read in
							const unsigned long size = buffer.getBufferSize();

							// Pointer to the data that we will iterate through
							TypeName *data = buffer.getMutableData();

							// The sample we read in to
							char streamBuffer[3];

							// For the number of samples, we want to read in and convert to floating point range..
							for (unsigned long i = 0; i < size; i++)
							{
								// Read the 3 bytes from the buffer
								m_stream.readDataFromStream(streamBuffer, 3);

								// Convert sample
								*data++ = (TypeName)((int)CEndian::convertThreeBytesToTwentyFourBitInt(streamBuffer) << 8) / (TypeName)0x7fffffff;
							}
						}
					break;
					default:
						return false;
					break;
				}

				//m_stream.setStreamEndianess(false);

				// Sucess!
				return true;
			}

//	===========================================================================

			/**
			 * Write the header information and from the format
			 * @param format The format to write in
			 * @retval bool True if wrote the header correctly, false otherwise
			 */
			virtual bool writeHeader(const typename IAudioFileFormat<TypeName>::SAudioFileFormat &format)
			{
				// Check open
				if (!m_stream.isStreamOpen())
				{
					return false;
				}

				const long numberOfBytes  = format.m_numberOfChannels * ((const long)format.m_bitDepth) / 8;
				const long bytesPerSample = numberOfBytes * (const long)format.m_sampleRate;
				const long sndSize		  = numberOfBytes * format.m_numberOfSamples + 8;
				const long totalSize	  = sndSize +
											8  /* SSND Chunk */ +
											10 /* 80 Bit SR */  +
											8  /* FORM Chunk */ +
											12 /* AIFF / COMM / Chunk */ +
											4  /* Num channels + bit depth */ +
											4  /* Num samples */ -
											8;


				// Stream the header information
				m_stream << 'F' << 'O' << 'R' << 'M'				// FORM header
						 << totalSize								// File size
						 << 'A' << 'I' << 'F' << 'F'				// AIFF header
						 << 'C' << 'O' << 'M' << 'M'				// COMM header
						 << (long)18								// Comm size				// 4
						 << (short)format.m_numberOfChannels		// Number of channels		// 2
						 << (unsigned long)format.m_numberOfSamples	// Sample frames			// 4
						 << (short)format.m_bitDepth;				// Bit depth				// 2

				// Write the 80 bit sample rate
				if (!write80BitSampleRate(m_stream, format.m_sampleRate))
				{
					return false;
				}

				// Stream the SSND header
				m_stream << 'S' << 'S' << 'N' << 'D'				// SSND header
						 << sndSize									// Size of chunk
						 << (unsigned long)0						// Offset 0
						 << (unsigned long)0;						// Block size 0

				return true;
			}

			/**
			 * Write the entire audio file to the provided audio buffer
			 * @param buffer The data buffer to store
			 * @retval bool True if written correctly, false otherwise
			 */
			virtual bool writeData(const TAudioBuffer< TypeName > &buffer)
			{
				switch(m_bitDepth)
				{
					case IAudioFileFormat<TypeName>::e_eightBit:
						{
							const TypeName *data = buffer.getData();
							for (unsigned long i = 0; i < buffer.getNumberOfChannels() * buffer.getNumberOfSamples(); i++)
							{
								//unsigned char sample = (unsigned char)(255.0 * ((data[i] + 1.0) * 0.5));
								m_stream << (unsigned char)(255.0 * ((data[i] + 1.0) * 0.5));//sample;
							}
						}
					break;
					case IAudioFileFormat<TypeName>::e_sixteenBit:
						{
							const TypeName *data = buffer.getData();
							for (unsigned long i = 0; i < buffer.getNumberOfChannels() * buffer.getNumberOfSamples(); i++)
							{
								//short sample = (short)(data[i] * 32768);
								m_stream << (short)(data[i] * 32768);//sample;
							}
						}
					break;
					case IAudioFileFormat<TypeName>::e_twentyFourBit:
						{
							const TypeName *data = buffer.getData();
							char samples[3];
							for (unsigned long i = 0; i < buffer.getNumberOfChannels() * buffer.getNumberOfSamples(); i++)
							{
								CEndian::convertTwentyFourBitIntToThreeBytes((int)(data[i] * (double)0x7fffffff) >> 8, samples);
								m_stream.writeDataToStream(samples, 3);
							}
						}
					break;
					default:
						return false;
					break;
				}

				// Done! :)
				return true;
			}

//	===========================================================================

		protected:

//	===========================================================================

			/**
			 * @struct SAiffFileFormat TAiffFileFormat.hpp
			 * @brief Storeage for the file version of the aiff
			 */
			struct SAiffFileFormat
			{
				// FORM chunk
				char m_formId[4];						/**< 'FORM' */
				long m_formSize;						/**< Size of the file */
				char m_aiffFormatId[4];					/**< 'AIFF' */

				// COMM chunk
				char m_commId[4];						/**< 'COMM' */
				long m_commSize;						/**< Size of the file */
				short m_numberOfChannels;				/**< Number of chanenls */
				unsigned long m_numberOfSampleFrames;	/**< Number of sample frames */
				short m_bitsPerSample;					/**< Number of bits in each sample (16 / 24 etc) */
				char m_sampleRateBuffer[10];			/**< 80 bit IEEE Standard 754 floating point number - 10 bytes that are then converted to the correct unsigned long value */
				unsigned long m_sampleRate;				/**< The sample rate -> This is not inside the comm chunk. It is created from the above buffer. Dont try and write to it */

				// SSND chunk
				char m_ssndId[4];						/**< 'SSND' */
				long m_ssndSize;						/**< Size of the chunk */
				unsigned long m_offset;					/**< Offset size */
				unsigned long m_blockSize;				/**< Block size */
			};

//	===========================================================================

			/**
			 * Is this a valid riff wave
			 * @param stream The stream to read
			 * @param diskFormat The format header to check
			 * @retval bool True if aiff wave file format
			 */
			static bool readFileHeader(CFileStream &stream, SAiffFileFormat &diskFormat)
			{
				// Check stream is open
				if (!stream.isStreamOpen())
				{
					return false;
				}

				// Read the riff header
				stream >> diskFormat.m_formId[0]
					   >> diskFormat.m_formId[1]
					   >> diskFormat.m_formId[2]
					   >> diskFormat.m_formId[3]
					   >> diskFormat.m_formSize
					   >> diskFormat.m_aiffFormatId[0]
					   >> diskFormat.m_aiffFormatId[1]
					   >> diskFormat.m_aiffFormatId[2]
					   >> diskFormat.m_aiffFormatId[3];

				// Is this actually a wave format file?
				if (!isValidAiffWaveChunk(diskFormat))
				{
					return false;
				}

				// REad the comm id
				stream >> diskFormat.m_commId[0]
					   >> diskFormat.m_commId[1]
					   >> diskFormat.m_commId[2]
					   >> diskFormat.m_commId[3]
					   >> diskFormat.m_commSize;

				// Check the comm chuink
				if (!isValidCommChunk(diskFormat))
				{
					return false;
				}

				// Read the channel information
				stream >> diskFormat.m_numberOfChannels
					   >> diskFormat.m_numberOfSampleFrames
					   >> diskFormat.m_bitsPerSample;

				// Read the sample rate format...
				read80BitSampleRate(stream, diskFormat);

				// Read the SSND format
				stream >> diskFormat.m_ssndId[0]
					   >> diskFormat.m_ssndId[1]
					   >> diskFormat.m_ssndId[2]
					   >> diskFormat.m_ssndId[3]
					   >> diskFormat.m_ssndSize
					   >> diskFormat.m_offset
					   >> diskFormat.m_blockSize;

				// Check its a valid SSND chunk...
				if (!isValidSSNDChunk(diskFormat))
				{
					return false;
				}

				// We are done!
				return true;
			}

			/**
			 * Is this a valid riff wave
			 * @param format The format header to check
			 * @retval bool True if riff wave file format
			 */
			static bool isValidAiffWaveChunk(const SAiffFileFormat &format)
			{
				return (format.m_formId[0] 		 == 'F' &&
						format.m_formId[1] 		 == 'O' &&
						format.m_formId[2] 		 == 'R' &&
						format.m_formId[3]		 == 'M' &&
						format.m_aiffFormatId[0] == 'A' &&
						format.m_aiffFormatId[1] == 'I' &&
						format.m_aiffFormatId[2] == 'F' &&
						format.m_aiffFormatId[3] == 'F');
			}

			/**
			 * Is this a valid format chunk
			 * @param format The format header to check
			 * @retval bool True if the format has a valid format chunk
			 */
			static bool isValidCommChunk(const SAiffFileFormat &format)
			{
				return (format.m_commId[0] == 'C' &&
						format.m_commId[1] == 'O' &&
						format.m_commId[2] == 'M' &&
						format.m_commId[3] == 'M');
			}

			/**
			 * Read the sample rate in to the buffer
			 * @param stream The stream to read
			 * @param diskFormat The format header to check
			 * @retval bool True if sample rate read properly
			 */
			static bool read80BitSampleRate(CFileStream &stream, SAiffFileFormat &diskFormat)
			{
				// Stream in the samples
				if (!stream.readDataFromStream(diskFormat.m_sampleRateBuffer, 10 * sizeof(char)))
				{
					return false;
				}

				// Setup some temporary variables
				unsigned long last    = 0;
				unsigned char *buffer = (unsigned char *)diskFormat.m_sampleRateBuffer;

				// Swap the bytes over
				CEndian::swapFourBytesIfSystemIsLittleEndian((unsigned char *)((unsigned long *)(buffer + 2)));

				// First convert
				diskFormat.m_sampleRate = foolToLong((unsigned long *)(buffer + 2));
				unsigned char exponent  = 30 - *(buffer + 1);

				// Work out the long value
				while (exponent--)
				{
					last = diskFormat.m_sampleRate;
					diskFormat.m_sampleRate >>= 1;
				}

				// Move on by one to avoid rounding error
				if (last & 0x00000001) diskFormat.m_sampleRate++;

				// done
				return true;
			}

			/**
			 * Write unsigned long to 80 bit sample rate
			 * @param stream The stream to write
			 * @param sampleRate The sampler ate
			 * @retval bool True if sample rate read properly
			 */
			static bool write80BitSampleRate(CFileStream &stream, const unsigned long sampleRate)
			{
				// Check its open
				if (!stream.isStreamOpen())
				{
					return false;
				}

				// Construct the buffer to write to
				unsigned char buff[10];
				memset(buff, 0, 10 * sizeof(char));

				// Store local
				unsigned long sr = sampleRate;

				// Get the SR
				get80BitFloatingPointNumber(buff, sr);

				// Stream out the sample rate
				return stream.writeDataToStream(buff, 10 * sizeof(char));
			}

			/**
			 * Get 80 bit serial code
			 * @param buffer The buffer to write in to
			 * @param value The sample rate
			 * @note Why oh why did apple choose this format :S
			 */
			static void get80BitFloatingPointNumber(unsigned char *buffer, unsigned long value)
			{
				// Setup some temporary variables
				double doubleValue			  = (double)value;
				int sign					  = 0;
				int exponent				  = 0;
				double mantissa				  = 0.0;
				double fullScaleMantissa	  = 0.0;
				unsigned long highBitMantissa = 0;
				unsigned long lowBitMantissa  = 0;

				// First check if the value is negative and iff we have a sign bit
				if (doubleValue < 0)
				{
					sign		 = 0x8000;
					doubleValue *= -1;
				}
				else
				{
					sign = 0;
				}

				// Now compute the exponent and mantissa
				if (doubleValue == 0)
				{
					exponent		= 0;
					highBitMantissa = 0;
					lowBitMantissa  = 0;
				}
				else
				{
					mantissa = frexp(doubleValue, &exponent);
					if ((exponent > 16384) || !(mantissa < 1))
					{
						// Number is infinite or NaN
						exponent		= sign | 0x7FFF;
						highBitMantissa = 0;
						lowBitMantissa  = 0;
					}
					else
					{
        				// Finite number
						exponent += 16382;
						if (exponent < 0)
						{
            				// Denormalized
							mantissa = ldexp(mantissa, exponent);
							exponent = 0;
						}
						exponent		 |= sign;
						mantissa		  = ldexp(mantissa, 32);
						fullScaleMantissa = floor(mantissa);
						highBitMantissa   = DOUBLE_TO_UNSIGNED_LONG(fullScaleMantissa);
						mantissa		  = ldexp(mantissa - fullScaleMantissa, 32);
						fullScaleMantissa = floor(mantissa);
						lowBitMantissa    = DOUBLE_TO_UNSIGNED_LONG(fullScaleMantissa);
					}
				}

				// Bit shift the buffer to the correct places
				buffer[0] = exponent >> 8;
				buffer[1] = exponent;
				buffer[2] = highBitMantissa >> 24;
				buffer[3] = highBitMantissa >> 16;
				buffer[4] = highBitMantissa >> 8;
				buffer[5] = highBitMantissa;
				buffer[6] = lowBitMantissa >> 24;
				buffer[7] = lowBitMantissa >> 16;
				buffer[8] = lowBitMantissa >> 8;
				buffer[9] = lowBitMantissa;
			}

			/**
			 * Is this a valid SSND chunk
			 * @param format The format header to check
			 * @retval bool True if the format has a valid data chunk
			 */
			static bool isValidSSNDChunk(const SAiffFileFormat &format)
			{
				return (format.m_ssndId[0] == 'S' &&
					    format.m_ssndId[1] == 'S' &&
					    format.m_ssndId[2] == 'N' &&
					    format.m_ssndId[3] == 'D');
			}

			/**
			 * Fool compiler to long swap
			 * @param ptr The unsigned char ptr
			 * @retval unsigned long the Unsigned long verison
			 */
			static unsigned long foolToLong(unsigned long *ptr)
			{
				return(*ptr);
			}

//	===========================================================================

			CFileStream m_stream;													/**< The file stream that we have */
			typename IAudioFileFormat<TypeName>::EBitDepth m_bitDepth;				/**< Bit depth of files being written */
		};
	}
}
#endif	// End Of TAiffFileFormat.hpp